/************************************************************************************************************\

Module Name:    LVideoOut.h

Description:    The LVideoOut module of the API provides the programming interface for the video output
                hardware and its associated overlay, cursor, and connected output devices.

References:     LVideoOut.doc Revision 2.5.

    Copyright (c) 2016, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LVIDEOOUT_H
#define INC_LVIDEOOUT_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LBuffer.h"
#include "LVideo.h"
#include "LPixelFormat.h"

typedef struct LVIDEOOUT_OBJECT*     LVideoOut_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                   O V E R V I E W
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

The video output module allows the user to use the data stored in the buffers in the local memory of SV2 to
produce a video output signal through a video encoder. This signal can then be sent to an enabled display
device for which the parameters were properly configured.

A Liberatus application may control the rate at which a video output signal is processed by using set buffer 
and field counters. The field counter is incremented while the CRTC is running and indicates whether the 
previous data buffer is released and available for other Liberatus operations. Interlaced video settings use 
two fields per frame, and non-interlaced and progressive video settings use only one field per frame. The 
set buffer counter is incremented each time a buffer is set.

In interlaced video settings, the CRTC alternates between top and bottom fields. If no new buffer is received
on time, it repeats the previous image or the previous two fields alternately. The CRTC should wait until it
receives two new fields before releasing the two previous fields.

Since most LVideoOut functions work in immediate mode, the Liberatus application can control each image sent.
However, function names containing the word "Schedule" correspond to functions that allow the user to push
ahead commands to be executed as synchronized operations in the Device Thread.
For example, LVideoOut_SetBuffer is an immediate mode function, but LVideoOut_ScheduleSetBuffer is a
"push ahead" function. Once the CRTC is stopped, the Liberatus application should flush the remaining
commands.

The field counter is incremented by multiples of 16 for each complete field. This increment is defined by
LVIDEO_FIELD_COUNTER_INCREMENT. The unit "1/16 of a field" may be used with the
LVideoOut_ScheduleWaitFieldCounter function to control and synchronize operations on parts of an image.
For example, the 1000th field corresponds to the number 16000 of the field counter. Half or more of this
field will be displayed after 16008 of the field counter. It will be fully displayed when the field counter
reaches 16016, which corresponds to the 1001th field. The LVideoOut_GetCounters function only returns full
image counts (multiples of 16).

The set buffer counter is incremented by 1 everytime a buffer is set.

The support of some functions depends on the product. The LVideoOut_GetCaps function can be used to
determine whether or not a product supports a function.

Video output uses the physical resources of the product. A handle must be created with LVideoOut_GetHandle
to control it. When requesting a handle, one of the three following access modes must be chosen: READWRITE,
READWRITE_EXCLUSIVE, or READONLY mode. These are provided by the LAccesMode object. The Exclusive mode
prevents an application to interfere with another application when configuring video outputs.
The Liberatus application can create as many READONLY handles as desired but can't call functions that change
the system state. In some cases, an application may have to create several LAccessMode_READWRITE handles to
manage the main video and overlay or cursor output in separate threads. The overlay and cursor are disabled
when the main display is disabled.


\************************************************************************************************************/


// -----------------------------------------------------------------------------------------------------------
//                                   U S A G E
// -----------------------------------------------------------------------------------------------------------

/*  Display Buffer on a Video output.

    #include "Liberatus.h"
    #include "LBuffer.h"
    #include "LVideoOut.h"

    int main()
    {
        LDevice_Handle              hDevice;
        LBuffer_Handle              hBuffer         = MNULL;
        LVideoOut_Handle            hVideoOut       = MNULL;
        LVideo_VidParam             oVidParam       = {0};
        LBuffer_VideoAttributes     oVideoAttribute = {0};

        Liberatus_Load();

        hDevice = Liberatus_GetDevice(0);           // Primary device

        LVideoOut_GetHandle(hDevice,                // Primary handle
                            0,                      // Output index
                            LAccessMode_READWRITE_EXCLUSIVE,
                            &hVideoOut);

        LVideoOut_GetOptimalVideoParameters(hVideoOut,
                                            LVideo_SignalType_USEDEFAULT,
                                            &oVidParam);

        oVideoAttribute.eAttributeType  = LBuffer_Type_VIDEO;
        oVideoAttribute.ePixelFormat    = LPixelFormat_B8G8R8A8;
        oVideoAttribute.uiWidth         = oVidParam.uiHActive;
        oVideoAttribute.uiHeight        = oVidParam.uiVActive;

        LBuffer_Create(hDevice,
                       &oVideoAttribute.eAttributeType,
                       &hBuffer);

        LVideoOut_Enable(hVideoOut,
                         LVideo_SignalType_USEDEFAULT,
                         &oVidParam,
                         hBuffer,
                         LVideoOut_PowerState_OFF);

        LVideoOut_SetPowerState(hVideoOut,
                                LVideoOut_PowerState_ON);

        // USE VIDEO OUTPUT HERE

        LVideoOut_Disable(hVideoOut);

        LVideoOut_ReleaseHandle(hVideoOut);

        LBuffer_Destroy(hBuffer);

        Liberatus_UnLoad();

        return 0;
    }

*/

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
/************************************************************************************************************\

Enum:           LVideoOut_OverlayState

Description:    Enumerates all possible overlay state values.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LVideoOut_OverlayState_DISABLED                  = 0x00000000,  // Overlay is disabled.
    LVideoOut_OverlayState_COLOR_KEY                 = 0x00000001,  // Overlay is enabled using
                                                                    //  LVideoOut_EnableOverlayWithColorKey
    LVideoOut_OverlayState_PIXEL_ALPHA_BLENDING      = 0x00000002,  // Overlay is enabled using
                                                                    //  LVideoOut_EnableOverlayWithPixelAlphaBlending
    LVideoOut_OverlayState_CONSTANT_BLENDING_FACTOR  = 0x00000003,  // Overlay is enabled using
                                                                    //  LVideoOut_EnableOverlayWithConstantBlendingFactor

    LVideoOut_OverlayState_FORCE32                   = 0x7FFFFFFF,  // Dummy value to force enumeration to
                                                                    // use 32 bits.
} LVideoOut_OverlayState;

/************************************************************************************************************\

Enum:           LVideoOut_OverlayColorKeyMode

Description:    Enumerates all possible overlay color key mode values.

Comments:       - Color key mode is used to determine which buffer, the primary or the overlay, is containing
                  the color keyed pixels.

\************************************************************************************************************/
typedef enum
{
    LVideoOut_OverlayColorKeyMode_PRIMARY       = 0x00000000,   // Color key buffer is primary video buffer.
    LVideoOut_OverlayColorKeyMode_OVERLAY       = 0x00000001,   // Color key buffer is overlay video buffer.

    // Aliases.
    LVideoOut_OverlayColorKeyMode_DESTINATION   = LVideoOut_OverlayColorKeyMode_PRIMARY,
    LVideoOut_OverlayColorKeyMode_SOURCE        = LVideoOut_OverlayColorKeyMode_OVERLAY,

    LVideoOut_OverlayColorKeyMode_FORCE32       = 0x7FFFFFFF,   // Dummy value to force enumeration to use
                                                                // 32 bits.
} LVideoOut_OverlayColorKeyMode;

/************************************************************************************************************\

Enum:           LVideoOut_OverlayAlphaMode

Description:    Enumerates all possible overlay alpha mode values.

Comments:       - Alpha mode is used to determine the alpha component of pixels in which buffer, the primary
                  or the overlay, is used.

\************************************************************************************************************/
typedef enum
{
    LVideoOut_OverlayAlphaMode_PRIMARY      = 0x00000000,   // Alpha of primary video buffer is used.
    LVideoOut_OverlayAlphaMode_OVERLAY      = 0x00000001,   // Alpha of overlay video buffer is used

    // Aliases.
    LVideoOut_OverlayAlphaMode_DESTINATION  = LVideoOut_OverlayAlphaMode_PRIMARY,
    LVideoOut_OverlayAlphaMode_SOURCE       = LVideoOut_OverlayAlphaMode_OVERLAY,

    LVideoOut_OverlayAlphaMode_FORCE32      = 0x7FFFFFFF,   // Dummy value to force enumeration to uses
                                                            // 32 bits.
} LVideoOut_OverlayAlphaMode;

/************************************************************************************************************\

Enum:           LVideoOut_PowerState

Description:    Enumerates all possible video output power states.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LVideoOut_PowerState_DISABLED      = 0x00000000,   // Output is disabled and not usable.
    LVideoOut_PowerState_ON            = 0x00000001,   // H-sync on,  V-sync on
    LVideoOut_PowerState_BLANKED       = 0x00000002,   // H-sync on,  V-sync on,  recovery Time ~0 second.
    LVideoOut_PowerState_STANDBY       = 0x00000004,   // H-sync off, V-sync on,  recovery Time ~1 second.
    LVideoOut_PowerState_SUSPEND       = 0x00000008,   // H-sync on,  V-sync off, recovery Time ~5 seconds.
    LVideoOut_PowerState_OFF           = 0x00000010,   // H-sync off, V-sync off, recovery Time >5 seconds.

    LVideoOut_PowerState_FORCE32       = 0x7FFFFFFF,   // Dummy value to force enumeration to use 32 bits.
} LVideoOut_PowerState;


/************************************************************************************************************\

Enum:           LVideoOut_FlipType

Description:    Enumerates all possible flip types.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LVideoOut_FlipType_NEXT_FIELD           = 0x00000000,   // Flip executed only when the next field will be
                                                            // displayed.
    LVideoOut_FlipType_NEXT_FIELD_WAIT      = 0x00000001,   // Flip executed only when the next field will be
                                                            // displayed.  The call is blocking until the
                                                            // flip is complete.

    LVideoOut_FlipType_FORCE32              = 0x7FFFFFFF,   // Dummy value to force enumeration to use
                                                            // 32 bits.
} LVideoOut_FlipType;


/************************************************************************************************************\

Enum:           LVideoOut_Event

Description:    Enumerates all possible video output events.

Comments:       LHandleVideoOut.h defines valid mask as 0xFFFF. So only 16-lsb are valid for events.

\************************************************************************************************************/
typedef enum
{
    LVideoOut_Event_INVALID                     = 0x00000000,   // Invalid value
    LVideoOut_Event_SIGNAL_TYPE_CHANGED         = 0x00000001,   // LVideoOut_GetSignalType
    LVideoOut_Event_VIDEO_PARAMETERS_CHANGED    = 0x00000002,   // LVideoOut_GetVideoParameters
    LVideoOut_Event_POWERSTATE_CHANGED          = 0x00000004,   // LVideoOut_GetPowerState
    LVideoOut_Event_BUFFER_ORIGIN_CHANGED       = 0x00000008,   // LVideoOut_GetBufferOrigin
    LVideoOut_Event_ACTIVE_REGION_CHANGED       = 0x00000010,   // LVideoOut_GetActiveRegion
    LVideoOut_Event_GAMMA_RAMP_CHANGED          = 0x00000020,   // LVideoOut_GetGammaRamp
    LVideoOut_Event_OVERLAY_CHANGED             = 0x00000040,   // LVideoOut_GetOverlayInfo
    LVideoOut_Event_OVERLAY_REGION_CHANGED      = 0x00000080,   // LVideoOut_GetOverlayRegions
    LVideoOut_Event_OVERLAY_GAMMA_RAMP_CHANGED  = 0x00000100,   // LVideoOut_GetOverlayGammaRamp
    LVideoOut_Event_CURSOR_VISIBILITY           = 0x00000200,   // LVideoOut_GetCursorVisibility
    LVideoOut_Event_BLANKED                     = 0x00000400,   // LVideoOut_GetBlanked.
    LVideoOut_Event_UNBLANKED                   = 0x00000800,   // LVideoOut_GetUnblanked.
    LVideoOut_Event_FORCE32                     = 0x7FFFFFFF    // Dummy value to force to use 32 bits.
} LVideoOut_Event;


/************************************************************************************************************\

Struct:         LVideoOut_OverlayInfo

Description:    Defines the overlay state and visibility.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVideoOut_OverlayInfo
{
    MBOOL32                 bIsVisible;         // Overlay is visible. 
    LVideoOut_OverlayState  eOverlayState;      // Describes the overlay state, enable or disable. 
} LVideoOut_OverlayInfo;



/************************************************************************************************************\

Struct:         LVideoOut_Color1024

Description:    Defines a 10 bits per components RGB color.

Comments:       - To convert an 8 bits per component color to a 10 bits per component one,
                  use the following formula:

                                      (8-bit value) x 0x3ff
                    (10-bit value) = ----------------------
                                               0xff

                - In 10 bits per component value, the 8 bits per component value 0x8035D3
                  would become 0x2013534E:

                               Convert 8-bit component values:
                                   Component 0: 0xD3 * 0x3FF / 0xFF = 0x34E
                                   Component 1: 0x35 * 0x3FF / 0xFF = 0x0D4
                                   Component 2: 0x80 * 0x3FF / 0xFF = 0x201

                               Combine converted component values:
                                   0x34E + (0x0D4 << 10) + (0x201 << 20) = 0x2013534E

\************************************************************************************************************/
typedef union tagLVideoOut_Color1024
{
    MUINT32 uiColor;                    // Packed color representation on 32 bits.
    struct
    {
        MUINT32 uiBlue          : 10;   // [0, 1023] Blue component color intensity value.
        MUINT32 uiGreen         : 10;   // [0, 1023] Green component color intensity value.
        MUINT32 uiRed           : 10;   // [0, 1023] Red component color intensity value.
        MUINT32 uiReserved1     : 2;    // Reserved for future use, must be set to 0.
    };
} LVideoOut_Color1024;

/************************************************************************************************************\

Struct:         LVideoOut_GammaRampEntry

Description:    Defines a gamma ramp array.

Comments:       - Gamma ramp entries have three 16-bit values, one for each component, used to correct,
                  change, or adjust the intensity of a pixel before the pixel is displayed.  A gamma ramp can
                  be used to correct monitor imperfections by making displayed pixels appear brighter or
                  darker than their actual intensity.
                - Each entry must have a 16-bit pixel component intensity value.  For example, if the caller
                  has a formula to compute gamma ramp pixel component values on 8 bits, use the following
                  formula to convert the 8-bit values to the required 16-bit value:

                                      (8-bit value) x 65535
                    (16-bit value) = ----------------------
                                                255
                - A gamma ramps array of 1024 entries ranging from 0 through 65535 can be calculated using the
                  following formula:

                       y = GammaRamp[x] = INT((65535 * ((x / 1023) ^ (1 / GammaFactor))) + 0.5)

                               y |
                           65535 |                    .
                                 |               .
                       output    |          .
                       pixels    |      .
                                 |   .
                                 | .
                                 |______________________
                                                       x input pixels
                                                     1023

                        // Example of gamma 2.2 correction for the red component.
                        aoGammaRampEntries[0].uiRed       = 0x0000;
                        aoGammaRampEntries[1].uiRed       = 0x0AF8;
                        aoGammaRampEntries[2].uiRed       = 0x0F08;
                        ...
                        aoGammaRampEntries[512].uiRed     = 0xBAE5;
                        aoGammaRampEntries[513].uiRed     = 0xBB0F;
                        aoGammaRampEntries[514].uiRed     = 0xBB3A;
                        ...
                        aoGammaRampEntries[1021].uiRed    = 0xFFC5;
                        aoGammaRampEntries[1022].uiRed    = 0xFFE2;
                        aoGammaRampEntries[1023].uiRed    = 0xFFFF;

\************************************************************************************************************/
typedef struct tagLVideoOut_GammaRampEntry
{
    MUINT16 uiRed;          // Red   component value.
    MUINT16 uiGreen;        // Green component value.
    MUINT16 uiBlue;         // Blue  component value.
    MUINT16 uiReserved1;    // Reserved, must be set to 0.
} LVideoOut_GammaRampEntry;

/************************************************************************************************************\

Struct:         LVideoOut_CursorPixel

Description:    Defines a color cursor pixel value.

Comments:       - A color cursor pixel is 32 bits wide with-bit organized in A7X1R8G8B8 format where:

                    +----+-------+----------------------------------------------+
                    | Ch | Bits  | Description                                  |
                    +----+-------+----------------------------------------------+
                    | R8 | 00:07 | 8-bit red component color intensity value.   |
                    | G8 | 08:15 | 8-bit green component color intensity value. |
                    | B8 | 16:23 | 8-bit blue component color intensity value.  |
                    | X1 | 24:24 | 1-bit XOR mode state value.                  |
                    | A7 | 25:31 | 7-bit alpha component value.                 |
                    +----+-------+----------------------------------------------+

                  The A7 and X1 component values define how a color cursor pixel is blended with underlying
                  buffer pixels to generate the resulting visible pixel.

                  A) X1 equal 0

                      XOR mode is disabled.  The alpha component controls the blending using the
                      following formula:

                          [Visible color] = ([Cursor color]     *      Alpha/127 +
                                            ([Underlying color] * (1 - Alpha/127)

                      Special values of alpha:
                          A7 = 0x00   => Cursor pixel is transparent.
                          A7 = 0x7F   => Cursor pixel is opaque.

                  B) X1 equal 1

                      XOR mode is enabled.  The alpha component isn't used.  The visible color is
                      the XOR result of the cursor color and the underlying color using the
                      following formula:

                          [Visible color] = [Cursor color] XOR [Underlying color].

                              0 XOR 0 = 0
                              0 XOR 1 = 1
                              1 XOR 0 = 1
                              1 XOR 1 = 0

                      Special values of cursor pixel:
                          X1 = 1, R8 = 0xFF, G8 = 0xFF, B8 = 0xFF => Cursor pixel is the inverse
                                                                     of the underlying color.
                          X1 = 1, R8 = 0x00, G8 = 0x00, B8 = 0x00 => Cursor pixel is transparent.

                      XOR mode is used for monochrome cursors.

\************************************************************************************************************/
typedef union tagLVideoOut_CursorPixel
{
    MUINT32 uiColor;                   // Packed cursor pixel representation on 32 bits.
    struct
    {
        MUINT32 uiRed           : 8;   // [0, 255] Red component color intensity value.
        MUINT32 uiGreen         : 8;   // [0, 255] Green component color intensity value.
        MUINT32 uiBlue          : 8;   // [0, 255] Blue component color intensity value.
        MUINT32 uiXOR           : 1;   // [0, 1]   XOR source.
        MUINT32 uiAlpha         : 7;   // [0, 127] Alpha (set to 0x7F when is not use).
    };
} LVideoOut_CursorPixel;

/************************************************************************************************************\

Struct:         LVideoOut_CursorShape

Description:    Structure defining a color cursor shape.

Comments:       - Maximum size allowed for a cursor shape is 64x64.
                - Cursor shape width versus pitch.
                  uiXSize:          The number of pixels composing a single line of the cursor shape.  It must
                                    be between 1 and 64.
                  uiPitch:          The number of pixels separating a pixel of a line from the corresponding
                                    pixel in the next line.

                  The pitch is always greater or equal to the width value.  Most of the time, both values will
                  correspond, but in some rare circumstances, it might be useful to have the pitch greater
                  than the width.

                  For example, if two shapes live side-by-side in a single buffer, the pitch can be used to
                  skip the unused shape information.

                  On a line where pixels are ranging from '0' through 'pitch - 1', the pixels '0' to
                  'width - 1' contain shape information while the pixels 'width' to 'pitch - 1' are ignored.

\************************************************************************************************************/
typedef struct tagLVideoOut_CursorShape
{
    MUINT32                 uiXSize;        // [1, 64] Width, in pixels, of cursor shape.
    MUINT32                 uiYSize;        // [1, 64] Height, in lines, of cursor shape.
    MUINT32                 uiPitch;        // Number of pixels between consecutive lines in the array of
                                            // cursor pixels.  See comments for more details.
    MUINT32                 uiHotSpotX;     // Offset along the X axis, in pixel, between the cursor shape
                                            // top-left corner and the cursor hot-spot X position.
    MUINT32                 uiHotSpotY;     // Offset along the Y axis in lines between the cursor shape
                                            // top-left corner and the cursor hot-spot Y position.
    LVideoOut_CursorPixel*  paoPixels;      // Pointer to an array of LVideoOut_CursorPixel elements
                                            // containing the cursor shape information to set.
                                            // The array must contain at least ('uiXSize' * 'uiPitch')
                                            // entries to avoid memory overruns.
} LVideoOut_CursorShape;

/************************************************************************************************************\

Struct:         LVideoOut_Caps

Description:    Structure defining video output capabilities.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVideoOut_Caps
{
    MUINT32     uiSize;                         // Size of LVideoOut_Capabilities structure.
    MUINT       bActiveRegion           : 1;    // Active region feature is supported.
    MUINT       bBorderColor            : 1;    // Border color feature is supported.
    MUINT       bGammaRamp              : 1;    // Gamma ramp feature is supported.
    MUINT       bWaitForVerticalBlank   : 1;    // LVideoOut_WaitForVerticalBlank is supported.
    MUINT       bFieldScanLineNumber    : 1;    // LVideoOut_GetFieldScanLineNumber is supported.
    MUINT       bOverlayColorKey        : 1;    // Overlay with color key feature is supported.
    MUINT       bOverlayAlphaBlending   : 1;    // Overlay with alpha blending feature is supported.
    MUINT       bOverlayConstantBlending: 1;    // Overlay with constant blending is supported.
    MUINT       bOverlayGammaRamp       : 1;    // Overlay gamma ramp feature is supported.
    MUINT       bCursor                 : 1;    // Cursor feature is supported.
    MUINT       bTickRefPixelClockAdjust: 1;    // Recovered a pixel clock with tick reference is supported.
    MUINT       bBufferOrigin           : 1;    // LVideoOut_SetBufferOrigin is supported and larger surfaces 
                                                //  than video parameters active display can be use for 
                                                //  LVideoOut_Enable.
    MUINT       bReserved1              : 20;   // Reserved for future use.
} LVideoOut_Caps;


// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LVideoOut_GetCount

Description:    Gets the number of independent video outputs available for the specified device.

Parameters:     IN  hDev            Device handle.
                OUT puiCount        Pointer to a MUINT32 variable where the number of independent video outputs
                                    available is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetCount(
                LDevice_Handle      hDev,
                MUINT32*            puiCount);

/************************************************************************************************************\

Function:       LVideoOut_GetCaps

Description:    Gets the video output capabilities 

Parameters:     IN      hDev                Device handle.
                IN      uiIndex             Index of the independent video output for which the caps should be 
                                             returned.
                IN OUT  poCapabilities      Pointer to an LVideoOut_Caps object where the video 
                                             capabilities are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Certain products may not support some video output features, and these functions will failed
                  with LStatus_UNSUPPORTED.
                - Fill LVideoOut_Caps->uiSize with the size of LVideoOut_Caps structure to use it.
                - LVideoOut_Caps->uiSize contains the size of the information returned.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetCaps(
                LDevice_Handle          hDev,
                MUINT32                 uiIndex,
                LVideoOut_Caps*         poCapabilities);

/************************************************************************************************************\

Function:       LVideoOut_GetHandle

Description:    Gets the video output handle for the specified independent video output.

Parameters:     IN  hDev            Device handle.
                IN  uiIndex         Index of the independent video output for which to return the handle.
                IN  eAccessMode     Requested access mode.  Must be a value from LAccessMode.
                OUT phVideoOut      Pointer to an LVideoOut_Handle variable where the video output device
                                    handle is written.  MNULL if unsuccessful.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Invalid video output device index.  (No hardware associated).
                LStatus_OUT_OF_RESOURCES    Specified video output device can't be used by the Liberatus
                                            applications because it's already being used by the system.
                LStatus_RESOURCES_BUSY      Video output handle couldn't be returned because the video output
                                            is busy.

Comments:       - If eAccessMode is LAccessMode_READONLY, functions that modify the video output, overlay, 
                  or cursor can't be called.
                - If eAccessMode is LAccessMode_READWRITE or LAccessMode_READWRITE_EXCLUSIVE, functions that 
                  modify the video output, overlay, or cursor can be called.
                - Only one handle with LAccessMode_READWRITE_EXCLUSIVE read/write access mode can exist at 
                  any one time. If a handle with read/write access has already been granted to a caller, 
                  any attempt to get another handle with read/write access returns a failure.
                - If eAccessMode is LAccessMode_READWRITE_EXCLUSIVE, the video output will be disabled when
                  the handle is released.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetHandle(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                LAccessMode         eAccessMode,
                LVideoOut_Handle*   phVideoOut);

/************************************************************************************************************\

Function:       LVideoOut_ReleaseHandle

Description:    Releases a handle to an independent video output.

Parameters:     IN hVideoOut            Handle to video output device to release.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - If eAccessMode is LAccessMode_READWRITE_EXCLUSIVE, the video output will be disabled when
                  the handle is released.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ReleaseHandle(
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LVideoOut_EnumSignalType

Description:    Enumerates all supported video output signal types.

Parameters:     IN  hVideoOut           Handle to video output device.
                IN  uiIndex             Index of the signal type to enumerate.
                OUT peSignalType        Pointer to an LVideo_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    End of the enumeration.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_EnumSignalType(
                LVideoOut_Handle        hVideoOut,
                MUINT32                 uiIndex,
                LVideo_SignalType*      peSignalType);

/************************************************************************************************************\

Function:       LVideoOut_GetDefaultSignalType

Description:    Gets the default signal type of the video output device.  

Parameters:     IN  hVideoOut           Handle to video output device.
                OUT peSignalType        Pointer to an LVideo_SignalType object where the default signal type
                                        is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The monitor may implicitly be detected when calling this function if it has not already
                  been detected by LBoard_DetectMonitor. The last monitor detection is used to respond
                  properly.
                - The default signal type may change after a new detection of the connected monitors.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetDefaultSignalType(
                LVideoOut_Handle    hVideoOut,
                LVideo_SignalType*  peSignalType);

/************************************************************************************************************\

Function:       LVideoOut_GetSignalType

Description:    Gets the current signal type of the video output device.  

Parameters:     IN  hVideoOut           Handle to video output device.
                OUT peSignalType        Pointer to an LVideo_SignalType object where the current signal type
                                        is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetSignalType(
                LVideoOut_Handle    hVideoOut,
                LVideo_SignalType*  peSignalType);


/************************************************************************************************************\

Function:       LVideoOut_GetOptimalVideoParameters

Description:    Gets the optimal video parameters for the specified signal type.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  eSignalType         Signal type for which to get the optimal video parameters.  To get the
                                        optimal video parameters for the default signal type, use
                                        LVideo_SignalType_USEDEFAULT.
                OUT poVidParam          Pointer to an LVideo_VidParam object where the optimal video
                                        parameters are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NOT_OPTIMAL     Default, possibly not optimal, video parameters were returned as no
                                        information was available to determine the optimal ones (See comments
                                        for more details).

Comments:       - The monitor may implicitly be detected calling this function if it has not already
                  been detected by LBoard_DetectMonitor. Alternatively, the cover of the last detection is
                  used to respond properly.
                - If no monitor is detected, or if the detected monitor returns no EDID information, the
                  default VESA mode of 1024x768@60Hz is returned as the default video parameter.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetOptimalVideoParameters(
                LVideoOut_Handle    hVideoOut,
                LVideo_SignalType   eSignalType,
                LVideo_VidParam*    poVidParam);

/************************************************************************************************************\

Function:       LVideoOut_EnumVideoParameters

Description:    Enumerates the recommended video parameters for the specified signal type.  

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  eSignalType         Video signal type for which to enumerate the video parameters.  Must
                                        be a value from LVideo_SignalType.
                IN  uiIndex             Index of the video parameters to enumerate.
                OUT poVidParam          Pointer to an LVideo_VidParam object where the enumerated video
                                        parameters are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No video parameters for the given signal type are available.

Comments:       - The monitor may implicitly be detected calling this function if it has not already
                  been detected by LBoard_DetectMonitor. Alternatively, the cover of the last detection is
                  used to respond properly.
                - Enumerated video parameters are the best timings for the specified video output type based
                  on the EDID information the detected monitor returns.
                - The function can be called in a loop while incrementing the video parameters index to build
                  a list of valid video parameters, depending on the video output type. Some hardware or
                  monitors may have a restricted list of supported video parameters.
                - For certain signal types, the video parameters may need additional information than just the
                  default video parameters.  To get this additional information for the SDI, LVDS, and TV
                  formats, use LVideoOut_EnumVideoParameters.

\************************************************************************************************************/
LAPI LStatus LVideoOut_EnumVideoParameters(
                LVideoOut_Handle    hVideoOut,
                LVideo_SignalType   eSignalType,
                MUINT32             uiIndex,
                LVideo_VidParam*    poVidParam);

/************************************************************************************************************\

Function:       LVideoOut_ValidateVideoParameters

Description:    Tests the videos parameters for a video output.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  eSignalType         Video signal  type for which to test the video parameters.
                IN  poVidParam          Pointer to an LVideo_VidParam object containing the video parameters
                                        to validate.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed because the hardware doesn't support the
                                        video parameters using the specified SignalType
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ValidateVideoParameters(
                LVideoOut_Handle    hVideoOut,
                LVideo_SignalType   eSignalType,
                LVideo_VidParam*    poVidParam);

/************************************************************************************************************\

Function:       LVideoOut_GetVideoParameters

Description:    Returns the currently active video parameters for a video output.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT poVidParam          Pointer to an LVideo_VidParam object where the currently active video
                                        parameters are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - If the video output hasn't been enabled using LVideoOut_Enable, there are no video
                  parameters, and LStatus_FAIL is returned.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetVideoParameters(
                LVideoOut_Handle    hVideoOut,
                LVideo_VidParam*    poVidParam);

/************************************************************************************************************\

Function:       LVideoOut_Enable

Description:    Enables the video output in the specified operation mode.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  eSignalType         Video signal type for which to configure the video parameters.
                IN  poVidParam          Pointer to an LVideo_VidParam object containing the video parameters.
                IN  hVidBuffer          Pointer to LBuffer_Handle identifying the buffer to set.
                IN  ePowerState         Output power state to set.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed because the hardware doesn't support the
                                        video parameters using the specified SignalType
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - If adjustments are needed before enabling video output, set the ePowerState parameter to
                  LVideoOut_PowerState_OFF to prevent video output from being enabled automatically.
                - When video output is enabled, the buffer origin is always set to (0, 0). This  corresponds
                  to the top-left corner of hVidBuffer.
                - Call LVideoOut_EnumPixelFormat to get the supported pixel formats for the video buffer.

\************************************************************************************************************/
LAPI LStatus LVideoOut_Enable(
                LVideoOut_Handle        hVideoOut,
                LVideo_SignalType       eSignalType,
                LVideo_VidParam*        poVidParam,
                LBuffer_Handle          hVidBuffer,
                LVideoOut_PowerState    ePowerState);

/************************************************************************************************************\

Function:       LVideoOut_Disable

Description:    Disables the CRTC, overlay and cursor of the video output.

Parameters:     IN hVideoOut            Handle of the video output object.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_Disable(
                LVideoOut_Handle        hVideoOut);


/************************************************************************************************************\

Function:       LVideoOut_SetPowerState

Description:    Sets the video output state.  Can be used to turn video output on or off, blank it, or set 
                DPMS.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  ePowerState         Output power state to set.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - When it stops receiving sync signals, the monitor enters power management state.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetPowerState(
                LVideoOut_Handle        hVideoOut,
                LVideoOut_PowerState    ePowerState);

/************************************************************************************************************\

Function:       LVideoOut_GetPowerState

Description:    Gets the video output state.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT pePowerState        Pointer to an LVideoOut_PowerState object where the video output power
                                        state is written.  Will be a value from the LVideoOut_PowerState.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - If no configuration is enabled on the video output, LVideoOut_PowerState is set to DISABLED.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetPowerState(
                LVideoOut_Handle        hVideoOut,
                LVideoOut_PowerState*   pePowerState);

/************************************************************************************************************\

Function:       LVideoOut_SetBufferOrigin

Description:    Sets the origin of the first pixel in the video buffer displayed by the video output.

Parameters:     IN hVideoOut            Handle of the video output object.
                IN uiX                  Origin of the first pixel on the x-axis.
                IN uiY                  Origin of the first pixel on the y-axis.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - This service is supported if bBufferOrigin of the LVideoOut_Caps structure is set.
                - Used when the video buffer is larger than the active video size (for example, when using
                  stretched mode or video panning).
                - The buffer origin can be changed dynamically.
                - If the specified origin combined with the active video size exceeds the video buffer limits,
                  LStatus_INVALID_PARAM is returned.
                - The origin override is always lost after calling LVideoOut_Enable, and it's always kept when
                  new video buffers are set using LVideoOut_SetBuffer and LVideoOut_ScheduleSetBuffer.

Example:
            The following shows an active video size of 1280x1024 using a single video buffer large enough
            to contain 4 times the number of pixels required by the active video size in a 2x2 array layout:

                |----------------2560---------------|
                |------1280-------|------1280-------|
                 _________________ _________________  ___   ___
                |                 |                 |  |     |
                |                 |                 |  |     |
                |     Viewport    |    Viewport     | 1024   |
                |        A        |        B        |  |     |
                |                 |                 |  |     |
                |_________________|_________________| _|_   2048
                |                 |                 |  |     |
                |                 |                 |  |     |
                |     Viewport    |    Viewport     | 1024   |
                |        C        |        D        |  |     |
                |                 |                 |  |     |
                |_________________|_________________| _|_   _|_

                Viewport A Origin = (0, 0)
                Viewport b Origin = (1280, 0)
                Viewport C Origin = (0, 1024)
                Viewport D Origin = (1280, 1024)

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetBufferOrigin(
                LVideoOut_Handle    hVideoOut,
                MUINT32             uiX,
                MUINT32             uiY);

/************************************************************************************************************\

Function:       LVideoOut_GetBufferOrigin

Description:    Gets the origin of the first pixel in the video buffer.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT puiX                Pointer to an MUINT32 variable where the origin, along the X axis, of
                                        the first pixel is written.
                OUT puiY                Pointer to an MUINT32 variable where the origin, along the Y axis, of
                                        the first pixel is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetBufferOrigin(
                LVideoOut_Handle    hVideoOut,
                MUINT32*            puiX,
                MUINT32*            puiY);

/************************************************************************************************************\

Function:       LVideoOut_SetActiveRegion

Description:    Sets the size and position of the active video output (source cropping).

Parameters:     IN hVideoOut            Handle of the video output object.
                IN poActiveRegion       Pointer to an LRECT32 object containing the active region to set.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - The default border color is black.  The color can be changed using LVideoOut_SetBorderColor.
                - This service is supported if bActiveRegion of the LVideoOut_Caps structure is set.
                - The active region can be changed dynamically.
                - LVideoOut_Enable sets the active region following the rules below:
                    1) Video buffer size >= video parameters size

                        Active region = (0,
                                         0,
                                         video parameters width,
                                         video parameters height).

                    2) Video buffer size < active video parameters size

                        Active region = (0,
                                         0,
                                         video buffer width,
                                         video buffer height).

                - An active region that's smaller than the video parameters creates a border region.  For more
                  information, see LVideo_VidParam.

Example :       To center a buffer of 1024 x 768 in a primary video output of 1280 x 1024:

                                                 VidParm 1280x1024
                                                   _______________________
                   Buffer 1024x768               |  1280                 |
                      _____________              |     _____________     |
                     | 1024        |             |    |   1024      |1024|
                     |          768|  ------->   |    |  Active  768|    |
                     |             |             |    |  region     |    |
                     |_____________|             |    |_____________|    |
                                                 | Border region         |
                                                 |_______________________|

                    poActiveRegion->iLeft   = 128;
                    poActiveRegion->iTop    = 128;
                    poActiveRegion->iRight  = poActiveRegion->iLeft + 1024;
                    poActiveRegion->iBottom = poActiveRegion->iTop  + 768;

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetActiveRegion(
                LVideoOut_Handle        hVideoOut,
                LRECT32*                poActiveRegion);

/************************************************************************************************************\

Function:       LVideoOut_GetActiveRegion

Description:    Gets the size and position of the active video output.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT poActiveRegion      Pointer to a LRECT32 object where the video output active region is
                                        written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - For more information, see LVideoOut_SetActiveRegion.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetActiveRegion(
                LVideoOut_Handle        hVideoOut,
                LRECT32*                poActiveRegion);

/************************************************************************************************************\

Function:       LVideoOut_SetBorderColor

Description:    Sets the color of the border region.

Parameters:     IN hVideoOut            Handle of the video output object.
                IN oBorderColor         Border color to set.
                IN bAllVisibleRegion    MTRUE to apply the border color to the border region and active
                                        region, including the video overlay.  MFALSE to apply the border color
                                        to the border region only.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bBorderColor of the LVideoOut_Caps structure is set.
                - This function can be used to dim the output or to set a specific color for image adjustment.
                - The default border color is black.
                - Applying a gamma ramp doesn't affect the border color.
                - If a color correction is needed, apply the correction to the border color before setting the
                  corrected color as the border color.
                - For more information, see LVideoOut_SetActiveRegion.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetBorderColor(
                LVideoOut_Handle        hVideoOut,
                LVideoOut_Color1024     oBorderColor,
                MBOOL32                 bAllVisibleRegion);

/************************************************************************************************************\

Function:       LVideoOut_GetBorderColor

Description:    Gets the color of the border region.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT poBorderColor       Pointer to an LVideoOut_Color1024 object where the border color is
                                        written.
                OUT pbAllVisibleRegion  Pointer to an MBOOL32 variable.  Set to MTRUE if the border color is
                                        also applied to the active region.  Otherwise, set to MFALSE.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bBorderColor of the LVideoOut_Caps structure is set.
                - The default border color is black.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetBorderColor(
                LVideoOut_Handle        hVideoOut,
                LVideoOut_Color1024*    poBorderColor,
                MBOOL32*                pbAllVisibleRegion);

/************************************************************************************************************\

Function:       LVideoOut_GetGammaRampInfo

Description:    Gets information about the video output gamma ramp.

Parameters:     IN  hVideoOut                   Handle of the video output object.
                OUT puiCurrentRampEntryCount    Pointer to an MUINT32 variable where the current gamma ramp
                                                array entry count is written.  For current gamma ramp
                                                information, this is the number of entries the array should
                                                have when passed to LVideoOut_GetGammaRamp.
                OUT puiOptimalRampEntryCount    Pointer to an MUINT32 variable where the optimal gamma ramp
                                                array entry count is written.  Used to avoid any mismatch
                                                between the applied gamma ramp and the gamma ramp used by the
                                                video output.  For optimal results, this is the number of
                                                entries the array should have when passed to
                                                LVideoOut_SetGammaRamp.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bGammaRamp of the LVideoOut_Caps structure is set.
                - puiOptimalRampEntryCount is dependent on the current configuration of the video output.
                  Because the current configuration isn't known until the video is enabled, call
                  LVideoOut_Enable before calling LVideoOut_GetGammaRampInfo.
                - For more information, see LVideoOut_GammaRampEntry.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetGammaRampInfo(
                LVideoOut_Handle    hVideoOut,
                MUINT32*            puiCurrentRampEntryCount,
                MUINT32*            puiOptimalRampEntryCount);

/************************************************************************************************************\

Function:       LVideoOut_SetGammaRamp

Description:    Sets the correction ramps for a video output. Used to correct the monitor gamma with a 
                hardware LUT.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  uiRampEntryCount    Number of entries in the paoRampEntries.  Set to 0 to restore the
                                        default linear ramp.
                IN  paoRampEntries      Pointer to a user-allocated array of LVideoOut_GammaRampEntry
                                        structures containing the gamma ramp values to set.   To avoid memory
                                        overruns, this parameter must have at least the same number of entries
                                        as in uiRampEntryCount.  Set to MNULL to restore the default linear
                                        ramp.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_NOT_OPTIMAL     uiRampEntryCount doesn't match the optimal gamma ramp.  paoRampEntries
                                        was modified to match the optimal gamma ramp size before being
                                        applied.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bGammaRamp of the LVideoOut_Caps structure is set.
                - When video output is enabled, the ramp is programmed directly in the hardware, typically
                  using a LUT.
                - If uiRampEntryCount has more entries than the optimal gamma ramp, only a subset, covering
                  the whole range described by paoRampEntries, of uiRampEntryCount will be used.
                - If uiRampEntryCount has fewer entries than the optimal gamma ramp, missing entries are
                  computed by interpolation.  For more information on the number of entries of an optimal 
                  gamma ramp, see LVideoOut_GetGammaRampInfo.
                - A typical gamma ramp array has 256 or 1024 entries.
                - The gamma ramp is applied before the pixels mixer of the video output.  Alpha component and
                  color key aren't affected by the correction gamma ramp.
                - The default linear ramps use the input pixel intensity as the output pixel intensity and is
                  applied when LVideoOut_Enable is called.  To set a non-linear gamma ramp, call
                  LVideoOut_Enable before calling LVideoOut_SetGammaRamp.
                - For more information, see LVideoOut_GammaRampEntry.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetGammaRamp(
                LVideoOut_Handle            hVideoOut,
                MUINT32                     uiRampEntryCount,
                LVideoOut_GammaRampEntry*   paoRampEntries);

/************************************************************************************************************\

Function:       LVideoOut_GetGammaRamp

Description:    Gets the gamma ramp data for the video output.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  uiRampEntryCount    Number of entries in paoRampEntries.
                IN  paoRampEntries      Pointer to a user-allocated array of LVideoOut_GammaRampEntry
                                        structures where the gamma ramp values are written.  To avoid memory
                                        overruns, this parameter must have at least the same number of entries
                                        as in uiRampEntryCount.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bGammaRamp of the LVideoOut_Caps structure is set.
                - If uiRampEntryCount is different from the current gamma ramp entry count,
                  LStatus_FAIL is returned.
                - For the correct entry count to use, see LVideoOut_GetGammaRampInfo.
                - For more information, see LVideoOut_GammaRampEntry.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetGammaRamp(
                LVideoOut_Handle            hVideoOut,
                MUINT32                     uiRampEntryCount,
                LVideoOut_GammaRampEntry*   paoRampEntries);


/************************************************************************************************************\

Function:       LVideoOut_WaitForVerticalBlank

Description:    Blocks a CPU thread until the CRTC enters the vertical blanking of the video parameters.  At
                this moment, the buffer is not requested.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  uiTimeout_ms        Waiting time in milliseconds.  0 means don't wait.
                                        LINFINITE_TIMEOUT means wait indefinitely.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a time out.
                LStatus_CANCELED        Function exited because it can't be completed.  (For example, the CRTC
                                        stopped running).
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bWaitForVerticalBlank of the LVideoOut_Caps structure is set.
                - For more information, see LVideo_VidParam.

\************************************************************************************************************/
LAPI LStatus LVideoOut_WaitForVerticalBlank(
                LVideoOut_Handle    hVideoOut,
                MUINT32             uiTimeout_ms);

/************************************************************************************************************\

Function:       LVideoOut_WaitForFieldStart

Description:    Blocks a CPU thread until the start of the next field

Parameters:     IN  hVideoOut               Handle of the video output object.
                IN  uiTimeout_ms            Waiting time in milliseconds.  0 means don't wait.
                                            LINFINITE_TIMEOUT means wait indefinitely.
                OUT puiFieldCounter         Pointer to an MUINT64 variable where the number of fields 
                                            displayed by the video output is written. MNULL if this counter 
                                            value isn't requested.
                OUT puiSetBufferCounter     Pointer to an MUINT64 variable where the number of set buffers
                                            performed by the video output is written. MNULL if this
                                            counter value isn't requested.
                OUT puiTickRefCounter       Pointer to an MUINT64 variable where the tick reference counter
                                            value is written. This value represent the tick number at which
                                            the field counter was updated. Ticks are incremented from the
                                            start of the system. MNULL if this counter value isn't requested.
                OUT pbIsTopField            Pointer to an MBOOL32 variable.  MTRUE if the top field is
                                            displayed.  MFALSE if the bottom field is displayed.  MNULL if the
                                            displayed field isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a time out.
                LStatus_CANCELED        Function exited because it can't be completed.  (For example, the CRTC
                                        stopped running).

Comments:       - Can be used for audio/video synchronization.
                - The field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.
                - If interlaced video parameters are used, use pbIsTopField to find out which of the top or
                  bottom fields are displayed.  If non-interlaced video parameters are used, pbIsTopField is
                  always set to MTRUE since there's no top or bottom field.
                - Progressive video parameters display only one field per frame.
                - The set buffer counter start at 1 and is increment each time LVideoOut_SetBuffer or 
                  LVideoOut_ScheduleSetBuffer is executed successfully.
                - The update of the set buffer counter is synchronized with the field counter and indicate what 
                  buffer is currently displayed.
                - A set buffer counter incremented by 2 indicate that a buffer was drop and was not 
                  displayed.

\************************************************************************************************************/
LAPI LStatus LVideoOut_WaitForFieldStart(
                LVideoOut_Handle    hVideoOut,
                MUINT32             uiTimeout_ms,
                MUINT64*            puiFieldCounter,
                MUINT64*            puiSetBufferCounter,
                MUINT64*            puiTickRefCounter,
                MBOOL32*            pbIsTopField);


/************************************************************************************************************\

Function:       LVideoOut_GetCounters

Description:    Gets the values of the set buffer and field counters.

Parameters:     IN  hVideoOut               Handle of the video output object.
                OUT puiFieldCounter         Pointer to an MUINT64 variable where the current number of fields
                                            displayed by the video output is written.  MNULL if this
                                            information isn't requested.
                OUT puiSetBufferCounter     Pointer to an MUINT64 variable where the number of set buffers
                                            performed by the video output is written. MNULL if this
                                            counter value isn't requested.
                OUT puiTickRefCounter       Pointer to an MUINT64 variable where the tick reference counter
                                            value is written. This value represents the tick number at which
                                            the field counter was updated. Ticks are incremented from the
                                            start of the system. MNULL if this counter value isn't requested.
                OUT pbIsTopField            Pointer to an MBOOL32 variable.  MTRUE if the top field is
                                            displayed.  MFALSE if the bottom field is displayed.  MNULL if
                                            displayed field isn't requested.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function can be used for audio/video synchronization.
                - The field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.
                - The field counter is incremented each time a field is displayed.
                - The overlay flip counter is incremented each time a new overlay buffer is used.
                - If interlaced video parameters are used, tw0 (2) fields and two (2) set buffers are 
                  required for a whole frame to be displayed. Use pbIsTopField to find out which of the top 
                  or bottom field are displayed. If non-interlaced video parameters are used, pbIsTopField 
                  is always set to MTRUE, since there's no top or bottom field.
                - Progressive video parameters display only one field per frame.     
                - The set buffer counter starts at 1 and is incremented each time LVideoOut_SetBuffer or 
                  LVideoOut_ScheduleSetBuffer is executed successfully.
                - The update of the set buffer counter is synchronized with the field counter and indicates  
                  which buffer is currently being displayed.
                - A set buffer counter incremented by 2 indicates that a buffer was dropped and not
                  displayed.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetCounters(
                LVideoOut_Handle    hVideoOut,
                MUINT64*            puiFieldCounter,
                MUINT64*            puiSetBufferCounter,
                MUINT64*            puiTickRefCounter,
                MBOOL32*            pbIsTopField);

/************************************************************************************************************\

Function:       LVideoOut_GetFieldCounterLUID

Description:    Gets the field counter unique identifiers (LUID).

Parameters:     IN  hVideoOut                   Handle of the video output object.
                OUT puiFieldCounterLUID         Pointer to an MUINT64 variable where the field counter's 
                                                unique identifier is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - This function can be used to synchronize the device thread with the completion of a counter.
                - The field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetFieldCounterLUID(
                LVideoOut_Handle    hVideoOut,
                MUINT64*            puiFieldCounterLUID);

/************************************************************************************************************\

Function:       LVideoOut_GetFieldScanLineNumber

Description:    Gets the currently scanned line number of the CRTC.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT puiScanLine         Pointer to an MUINT32 variable where the current field scan line value
                                        is written.  MNULL if this information isn't requested.
                OUT pbIsTopField        Pointer to an MBOOL32 variable.  MTRUE if the top field is displayed.
                                        MFALSE if the bottom field is displayed.  MNULL if this information
                                        isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bFieldScanLineNumber of the LVideoOut_Caps structure is set.
                - If the video output isn't enabled, LStatus_FAIL is returned.
                - If interlaced video parameters are used, use pbIsTopField to find out which of the top or
                  bottom field is displayed.
                - If non-interlaced video parameters are used, pbIsTopField is always set to MTRUE, since
                  there's no top or bottom field.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetFieldScanLineNumber(
                LVideoOut_Handle    hVideoOut,
                MUINT32*            puiScanLine,
                MBOOL32*            pbIsTopField);

/************************************************************************************************************\

Function:       LVideoOut_SetBuffer

Description:    Sets a new video buffer to display.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  hVideoBuffer        Handle to the new video buffer to display.
                IN  bIsTopField         MTRUE if the video buffer to display is a top field.
                                        MFALSE if the video buffer to display is a bottom field.
                IN  eFlipType           Type of flip to perform.  Must be a value from LVideoOut_FlipType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_CANCELED        The blocking process may no be achieve.  The VideoOut may not running.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - LVideoOut_SetBuffer overwrites any previously buffered set not yet displayed.
                - To enable video output, the new video buffer must have the same size and format as the video 
                  buffer supplied when LVideoOut_Enable was called.
                - If non-interlaced video parameters are used, bIsTopField is ignored since there's no top
                  or bottom field.
                - If the flip type is set to LVideoOut_FlipType_NEXT_FIELD, the flip is posted and
                  LVideoOut_SetBuffer returns immediately.  To find out when the new buffer is displayed, use
                  LVideoOut_WaitForFieldStart.
                - If the flip type is set to LVideoOut_FlipType_NEXT_FIELD_WAIT, the flip is posted and
                  LVideoOut_SetBuffer don't return until the newly set buffer is used.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetBuffer(
                LVideoOut_Handle    hVideoOut,
                LBuffer_Handle      hVideoBuffer,
                MBOOL32             bIsTopField,
                LVideoOut_FlipType  eFlipType);

/************************************************************************************************************\

Function:       LVideoOut_ScheduleSetBuffer

Description:    Schedules an update of the displayed video buffer by adding the required commands to
                hDeviceThread.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  hVideoBuffer        Handle of the video buffer to schedule.
                IN  bIsTopField         MTRUE if the video buffer to display is a top field.
                                        MFALSE if the video buffer to display is a bottom field.
                IN  hDeviceThread       Handle of the device thread used to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  eFlipType           Type of flip to perform.  Must be a value from LVideoOut_FlipType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.
                
Comments:       - See LVideoOut_SetBuffer function for more details.
                - LStatus_TIMEOUT is returned when the SetBuffer command couldn't be added to the device
                  thread because it was full and no room became available before the timeout period expired.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ScheduleSetBuffer(
                LVideoOut_Handle        hVideoOut,
                LBuffer_Handle          hVideoBuffer,
                MBOOL32                 bIsTopField,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                LVideoOut_FlipType      eFlipType);

/************************************************************************************************************\

Function:       LVideoOut_ScheduleWaitFieldCounter

Description:    Schedules a wait on a field counter of the video output device by adding the required 
                commands to the passed in device thread object.

Parameters:     IN  hVideoOut           Handle to the video output device.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  uiFieldCounter      Field counter value when the buffer will be displayed by 
                                        the video output.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_CANCELED        Function exited because the event will never occur.
                LStatus_TIMEOUT         Function exited after a time out.

Comments:       - LStatus_TIMEOUT is returned when the wait command couldn't be added to the device thread
                  because it was full and no room became available before the timeout period expired.
                - The field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ScheduleWaitFieldCounter(
                LVideoOut_Handle        hVideoOut,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MUINT64                 uiFieldCounter);

/************************************************************************************************************\

Function:       LVideoOut_SetBufferDelay

Description:    Delays the set buffer until the system reach the passed in field tick reference count.

Parameters:     IN  hVideoOut               Handle of the video output object.
                IN  uiFieldTickRefCounter   Field tick counter value when the buffer will idealy be displayed
                                            by the video output.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - This service is supported if bTickRefPixelClockAdjust of the LVideoOut_Caps structure is set.
                - Use this service to synchronize the start of the video.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetBufferDelay(
                LVideoOut_Handle    hVideoOut,
                MUINT64             uiFieldTickRefCounter);

/************************************************************************************************************\

Function:       LVideoOut_SetDesiredNextFieldTickReference

Description:    Sets desired next field tick reference.

Parameters:     IN  hVideoOut               Handle of the video output object.
                IN  uiFieldTickRefCounter   Field tick counter value when the buffer will ideally be displayed
                                            by the video output.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - This service is supported if bTickRefPixelClockAdjust of the LVideoOut_Caps structure is set.
                - Use to rebuild the ideal pixel clock.
                - Only one tick field counter should be set for every field.
                - When the next field tick counter is less than the effective field tick counter, the pixel
                  clock will speed-down smoothly.
                - When the next field tick counter is greater than the effective field tick counter, the pixel
                  clock will speed-up smoothly.
                - When the application stop to assist the field clock for every field, the nominal pixel clock
                  will smoothly be recover.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetDesiredNextFieldTickReference(
                LVideoOut_Handle    hVideoOut,
                MUINT64             uiFieldTickRefCounter);

/************************************************************************************************************\

Function:       LVideoOut_ScheduleSetDesiredNextFieldTickReference

Description:    Schedule the desired next field tick reference.

Parameters:     IN  hVideoOut               Handle of the video output object.
                IN  hDeviceThread           Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms            Device thread full queue wait timeout.
                IN  uiFieldTickRefCounter   Field tick counter value when the buffer will ideally be displayed
                                            by the video output.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a time out.

Comments:       - For more information, see the LVideoOut_SetDesiredNextFieldTickReference function.
                - LStatus_TIMEOUT is returned when the AdjustFieldClock command couldn't be added to the
                  device thread because it was full and no room became available before the timeout period
                  expired.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ScheduleSetDesiredNextFieldTickReference(
                LVideoOut_Handle        hVideoOut,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MUINT64                 uiFieldTickRefCounter);


/************************************************************************************************************\

    The Matrox Video Output module explains how to enable and customize a video output overlay.

    When certain algorithm criteria are met, the video overlay displays the secondary or overlay video buffer
    pixels instead of the primary video buffer pixels. You can configure the algorithms that determine whether
    to display the primary or the overlay pixels.

    Using video overlay on a video output requires at least two (2) video buffers:
       1. An overlay video buffer containing the pixels the video output will display when the overlay 
          algorithm criteria are met.
       2. A primary video buffer containing the pixels the video output will display when the overlay 
          algorithm criteria aren't met.

    Overlay algorithms use a region in both the primary and overlay video buffers to determine on which
    portion of the video buffers to apply the algorithms. The region in the primary video buffer is called
    the Overlay region, and the region in the overlay video buffer is called the Cropping region.

    The overlay and cropping regions must have the same width and height. If the width and/or height of the
    regions are different, scaling operations must be performed to make them identical before the regions can
    be used.

    For example, the following shows an overlay video buffer of 1280 x 1024 with a cropping region
    of 1024 x 768, and a primary video buffer of 1600 x 1200 with an overlay region of 1024 x 768:


     Overlay video buffer (Source)              Primary video buffer (Destination)
                 1280                                              1600
    +-----------------------------+             +----------------------------------------+
    |                             |             |                                        |
    |                             |             |                                        |
    |            1024             |             |                                        |
    |      +-----------------+    | 1           |                                        |
    |      | Source cropping |    | 0           |                                        |
    |      |   region        | 7  | 2           |                                        | 1
    |      |                 | 6  | 4           |                       1024             | 2
    |      |                 | 8  |             |                 +-----------------+    | 0
    |      |                 |    |             |                 | Overlay region  |    | 0
    |      +-----------------+    |             |                 |                 | 7  |
    +-----------------------------+             |                 |                 | 6  |
                                                |                 |                 | 8  |
                                                |                 |                 |    |
                                                |                 +-----------------+    |
                                                +----------------------------------------+

\************************************************************************************************************/

/************************************************************************************************************\

Function:       LVideoOut_EnableOverlayWithColorKey

Description:    Enables an overlay video configuration using a color keying algorithm for the video output.

Parameters:     IN hVideoOut                        Handle of the video output object.
                IN oColorKeyLowLimit                LVideoOut_Color1024 object defining the lower limit of the
                                                    color key range.
                IN oColorKeyHighLimit               LVideoOut_Color1024 object defining the upper limit of the
                                                    color key range.
                IN poOverlayRegion                  Pointer to an LRECT32 object defining the overlay region.
                IN poSourceCroppingRegionPosition   Pointer to an LPOS32 object defining the position of the
                                                    cropping region.  MNULL if the default position (0, 0)
                                                    should be used.
                IN hVideoBuffer                     Handle of the overlay video buffer.
                IN eColorKeyMode                    Specifies which video buffer contains the color keyed
                                                    pixels.  Must be a value from
                                                    LVideoOut_OverlayColorKeyMode.
                IN bShowOverlay                     MTRUE to immediately show the overlay.  MFALSE to enable
                                                    it without immediately showing it.  If MFALSE, color
                                                    keying pixels may be visible.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayColorKey of the LVideoOut_Caps structure is set.
                - The eColorKeyMode video buffer is used as a color keying buffer.  The color keying buffer
                  contains the pixels set to a color within the color key range.
                - When a primary video buffer pixel in the overlay region is about to be displayed, the color
                  of the primary video buffer pixel is compared to the color of the corresponding pixel in the
                  color keyed buffer.  If the primary video buffer pixel color is within the defined color key
                  range, the primary video buffer pixel is displayed.  Otherwise, the color keyed video buffer
                  pixel is displayed.
                - LVideoOut_EnableOverlayWithColorKey doesn't support scaling.
                - The pixel format of the overlay video buffer may be different from the pixel format of the
                  primary video buffer.  To be valid, the overlay video buffer size must be equal to or larger
                  than the size of the overlay region.
                - The color key range always uses 10 bits per component color values.
                - If a single color key value is used, specify the same value as both the low and high limits.
                - Call LVideoOut_Enable before calling LVideoOut_EnableOverlayWithColorKey.
                - Call LVideoOut_EnumPixelFormat to get the supported pixel formats.


\************************************************************************************************************/
LAPI LStatus LVideoOut_EnableOverlayWithColorKey(
                LVideoOut_Handle                hVideoOut,
                LVideoOut_Color1024             oColorKeyLowLimit,
                LVideoOut_Color1024             oColorKeyHighLimit,
                LRECT32*                        poOverlayRegion,
                LPOS32*                         poSourceCroppingRegionPosition,
                LBuffer_Handle                  hVideoBuffer,
                LVideoOut_OverlayColorKeyMode   eColorKeyMode,
                MBOOL32                         bShowOverlay);

/************************************************************************************************************\

Function:       LVideoOut_EnableOverlayWithPixelAlphaBlending

Description:    Enables an overlay video configuration using a per pixel alpha blending algorithm for the
                video output.

Parameters:     IN  hVideoOut                       Handle of the video output object.
                IN  poOverlayRegion                 Pointer to an LRECT32 object defining the overlay region.
                IN  poSourceCroppingRegionPosition  Pointer to an LPOS32 object defining the position of the
                                                    cropping region.  MNULL if the default (0, 0) position
                                                    should used.
                IN  hVideoBuffer                    Handle of the overlay video buffer.
                IN  eAlphaMode                      Specifies which video buffer pixel alpha component is used
                                                    to get the blending factor.  Must be a value from
                                                    LVideoOut_OverlayAlphaMode.
                IN  bShowOverlay                    MTRUE to immediately show the overlay.  MFALSE to enable
                                                    it without immediately showing it.  If MFALSE, color
                                                    keying pixels may be visible.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayAlphaBlending of the LVideoOut_Caps structure is set.
                - When a primary video buffer pixel in the overlay region is about to be displayed, the
                  primary video buffer pixel is compared to the corresponding pixel in the eAlphaMode buffers
                  to get the alpha blending factor.  The displayed pixel is the result of the blending of the
                  overlay and primary video buffer pixel values based on the blending factor.
                - LVideoOut_EnableOverlayWithPixelAlphaBlending doesn't support scaling.
                - The overlay video buffer pixel format may be different from the primary video buffer pixel
                  format.  To be valid, the overlay video buffer size must be equal to or larger than the
                  overlay region.
                - Call LVideoOut_Enable before calling LVideoOut_EnableOverlayWithPixelAlphaBlending.
                - Call LVideoOut_EnumPixelFormat to get the supported pixel formats.
                - The following formula can be used to predict the displayed pixel value:

                    if ( eAlphaMode == LVideoOut_OverlayAlphaMode_PRIMARY )
                    {
                        Alpha value   = Primary video buffer pixel alpha component value.
                        Pixel         = (Alpha value x Primary video buffer pixel) +
                                        ((1 - Alpha value) x Overlay video buffer pixel)
                    }
                    else if (eAlphaMode==LVideoOut_OverlayAlphaMode_OVERLAY)
                    {
                        Alpha value   = Overlay video buffer pixel alpha component value.
                        Pixel         = ((1 - Alpha value) x Primary video buffer pixel) +
                                        (Alpha value x Overlay video buffer pixel)
                    }

\************************************************************************************************************/
LAPI LStatus LVideoOut_EnableOverlayWithPixelAlphaBlending(
                LVideoOut_Handle            hVideoOut,
                LRECT32*                    poOverlayRegion,
                LPOS32*                     poSourceCroppingRegionPosition,
                LBuffer_Handle              hVideoBuffer,
                LVideoOut_OverlayAlphaMode  eAlphaMode,
                MBOOL32                     bShowOverlay);

/************************************************************************************************************\

Function:       LVideoOut_EnableOverlayWithConstantBlendingFactor

Description:    Enables an overlay video configuration using a constant blending factor algorithm for the
                video output.

Parameters:     IN  hVideoOut                       Handle of the video output object.
                IN  uiBlendingFactor                Blending factor.  Valid range: [0, 255].
                IN  poOverlayRegion                 Pointer to an LRECT32 object defining the overlay region.
                IN  poSourceCroppingRegionPosition  Pointer to an LPOS32 object defining the position of the
                                                    cropping region.  MNULL if default (0, 0) position should
                                                    used.
                IN  hVideoBuffer                    Handle of the overlay video buffer.
                IN  bShowOverlay                    MTRUE to immediately show the overlay.  MFALSE to enable
                                                    it without immediately showing it.  If not enabled, color
                                                    keying pixels may be visible.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - When a primary video buffer pixel in the overlay region is about to be displayed, its value
                  is blended with the value of the corresponding overlay video buffer pixel to get the
                  displayed pixel value.  All pixels in the overlay region are blended using the same blending
                  factor.
                - LVideoOut_EnableOverlayWithConstantBlendingFactor doesn't support scaling.
                - The overlay video buffer pixel format may be different from the primary video buffer pixel
                  format.  To be valid, the overlay video buffer size must be equal to or larger than the
                  overlay region.
                - Call LVideoOut_Enable before calling LVideoOut_EnableOverlayWithConstantBlendingFactor.
                - Call LVideoOut_EnumPixelFormat to get the supported pixel formats.
                - The following is a sample blending factor:
                     Pixel = (Blend Factor x primary video buffer pixel) +
                             ((1 - Blend Factor) x overlay video buffer pixel)

\************************************************************************************************************/
LAPI LStatus LVideoOut_EnableOverlayWithConstantBlendingFactor(
                LVideoOut_Handle    hVideoOut,
                MUINT32             uiBlendingFactor,
                LRECT32*            poOverlayRegion,
                LPOS32*             poSourceCroppingRegionPosition,
                LBuffer_Handle      hVideoBuffer,
                MBOOL32             bShowOverlay);

/************************************************************************************************************\

Function:       LVideoOut_GetOverlayInfo

Description:    Gets the information of the overlay.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT poOverlayInfo       Pointer to an poOverlayInfo object where the overlay information 
                                        is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayColorKey, bOverlayAlphaBlending or 
                  bOverlayConstantBlending, of the LVideoOut_Caps structure is set.
\************************************************************************************************************/
LAPI LStatus LVideoOut_GetOverlayInfo(
                LVideoOut_Handle        hVideoOut,
                LVideoOut_OverlayInfo*  poOverlayInfo);

/************************************************************************************************************\

Function:       LVideoOut_SetOverlayConstantBlendingFactor

Description:    Sets the overlay constant blending factor.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  uiBlendingFactor    New blending factor to set.  Valid range: [0, 255].

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - The new blending factor takes effect immediately, not when the next field is displayed.
                - The function fails if the overlay configuration wasn't enabled using
                  LVideoOut_EnableOverlayWithConstantBlendingFactor or
                  LVideoOut_SetOverlayConstantBlendingFactor services.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetOverlayConstantBlendingFactor(
                LVideoOut_Handle    hVideoOut,
                MUINT32             uiBlendingFactor);

/************************************************************************************************************\

Function:       LVideoOut_DisableOverlay

Description:    Disables the overlay configuration.

Parameters:     IN hVideoOut            Handle of the video output object.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - This service is supported if bOverlayColorKey, bOverlayAlphaBlending or 
                  bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - Overlay configuration information is lost when the overlay is disabled.  To re-enable a
                  disabled overlay, call one of the overlay enable functions.

\************************************************************************************************************/
LAPI LStatus LVideoOut_DisableOverlay(
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LVideoOut_SetOverlayRegions

Description:    Sets the positions of the overlay and cropping regions.

Parameters:     IN  hVideoOut                       Handle of the video output object.
                IN  poOverlayRegion                 Pointer to an LRECT32 object containing the new overlay
                                                    region to set.  MNULL if this information shouldn't be 
                                                    set.
                IN  poSourceCroppingRegionPosition  Pointer to an LPOS32 object containing the new cropping
                                                    region position to set.  MNULL if this information
                                                    shouldn't be set.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayColorKey, bOverlayAlphaBlending or 
                  bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - The positions of the overlay and cropping regions are relative to the upper-left corner of
                  their corresponding video buffer.  New overlay regions will be active only when the next
                  field is displayed.
                - The overlay region is the same size as the cropping region.  Only the position in the
                  overlay buffer can be changed.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetOverlayRegions(
                LVideoOut_Handle    hVideoOut,
                LRECT32*            poOverlayRegion,
                LPOS32*             poSourceCroppingRegionPosition);

/************************************************************************************************************\

Function:       LVideoOut_GetOverlayRegions

Description:    Gets the positions of the overlay and cropping regions.

Parameters:     IN  hVideoOut                       Handle of the video output object.
                OUT poOverlayRegion                 Pointer to an LRECT32 object where the overlay region
                                                    position is written.  MNULL if this information isn't
                                                    requested.
                OUT poSourceCroppingRegionPosition  Pointer to an LPOS32 object where the position of the
                                                    cropping region is written.  MNULL if this information
                                                    isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayColorKey, bOverlayAlphaBlending or 
                   bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - The positions of the overlay and cropping regions are relative to the upper-left corner of
                  their corresponding video buffer.
                - The overlay region is the same size as the.cropping region.  Only the position in the
                  overlay buffer can be changed.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetOverlayRegions(
                LVideoOut_Handle    hVideoOut,
                LRECT32*            poOverlayRegion,
                LPOS32*             poSourceCroppingRegionPosition);

/************************************************************************************************************\

Function:       LVideoOut_ShowOverlay

Description:    Shows the overlay.

Parameters:     IN hVideoOut            Handle of the video output object.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayColorKey, bOverlayAlphaBlending or 
                   bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - The overlay configuration is needed for the overlay to be visible.
                - Call LVideoOut_ShowOverlay only after calling one of the following:
                    - LVideoOut_EnableOverlayConfigWithConstantBlendingFactor
                    - LVideoOut_EnableOverlayConfigWithPixelAlphaBlendin
                    - LVideoOut_EnableOverlayConfigWithColorKey

\************************************************************************************************************/
LAPI LStatus LVideoOut_ShowOverlay(
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LVideoOut_HideOverlay.

Description:    Hides the overlay.

Parameters:     IN hVideoOut            Handle of the video output object.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayColorKey, bOverlayAlphaBlending or 
                   bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - The overlay configuration is kept while the overlay is hidden.  The overlay is hidden when
                  LVideoOut_Disable or LVideoOut_DisableOverlay are called.

\************************************************************************************************************/
LAPI LStatus LVideoOut_HideOverlay(
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LVideoOut_GetOverlayGammaRampInfo

Description:    Gets information about the video output overlay gamma ramp.

Parameters:     IN  hVideoOut                   Handle of the video output object.
                OUT puiCurrentRampEntryCount    Pointer to an MUINT32 variable where the current gamma ramp
                                                array entry count is written.  For current gamma ramp
                                                information, this is the number of entries the array should
                                                have when passed to LVideoOut_GetOverlayGammaRamp.
                OUT puiOptimalRampEntryCount    Pointer to an MUINT32 variable where the optimal gamma ramp
                                                array entry count is written.  This is used to avoid any
                                                mismatch between the applied gamma ramp and the one used by
                                                the video output.  For optimal results, this is the number of
                                                entries the array should have when passed to
                                                LVideoOut_SetOverlayGammaRamp.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayGammaRamp of the LVideoOut_Caps structure is set.
                - puiOptimalRampEntryCount is dependent on the current overlay configuration.
                - Call LVideoOut_GetOverlayGammaRampInfo only after calling one of the following:
                    - LVideoOut_EnableOverlayConfigWithConstantBlendingFactor
                    - LVideoOut_EnableOverlayConfigWithPixelAlphaBlendin
                    - LVideoOut_EnableOverlayConfigWithColorKey
                - The overlay gamma ramp is used to correct the intensity of the overlay displayed pixels.
                - For more information on gamma ramps, see LVideoOut_GammaRampEntry.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetOverlayGammaRampInfo(
                LVideoOut_Handle    hVideoOut,
                MUINT32*            puiCurrentRampEntryCount,
                MUINT32*            puiOptimalRampEntryCount);

/************************************************************************************************************\

Function:       LVideoOut_SetOverlayGammaRamp

Description:    Sets the correction ramps for a video output overlay.  Used to correct the monitor gamma using
                a hardware LUT.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  uiRampEntryCount    Number of entries in the paoRampEntries.  Set to 0 to restore the
                                        default linear ramp.
                IN  paoRampEntries      Pointer to a user-allocated array of LVideoOut_GammaRampEntry
                                        structures containing the gamma ramp values to set.  To avoid memory
                                        overruns, this parameter must have at least the same number of entries
                                        as in uiRampEntryCount.  Set to MNULL to restore the default linear
                                        ramp.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_NOT_OPTIMAL     The uiRampEntryCount doesn't match the optimal gamma ramp entry count
                                        for the current overlay configuration.  paoRampEntries was modified to
                                        match the optimal gamma ramp size before being applied.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayGammaRamp of the LVideoOut_Caps structure is set.
                - When video output overlay is enabled, the gamma ramp is programmed directly in the hardware
                  using a LUT.  The overlay gamma ramp is used to correct the intensity of the overlay
                  displayed pixels.
                - If the uiRampEntryCount has more entries than the optimal gamma ramp, only a subset covering
                  the whole range described by paoRampEntries gamma ramp entries, from of uiRampEntryCount the
                  passed in gamma ramp will be used.
                - If uiRampEntryCount has fewer entries than the optimal gamma ramp, the missing entries will
                  be computed by interpolation.  To find the number of entries for an optimal gamma ramp, see
                  LVideoOut_GetOverlayGammaRampInfo.
                - A typical gamma ramp array has 256 or 1024 entries.
                - The gamma ramp is applied before the pixels mixer.  Alpha component or color key aren't
                  affected by the correction gamma ramp.
                - The default linear ramps use the input pixel intensity for the output pixel intensity and
                  are applied when a video output overlay is enabled.  To set a non-linear gamma ramp, call
                  LVideoOut_SetOverlayGammaRamp only after calling one of the following:
                    - LVideoOut_EnableOverlayConfigWithConstantBlendingFactor
                    - LVideoOut_EnableOverlayConfigWithPixelAlphaBlendin
                    - LVideoOut_EnableOverlayConfigWithColorKey
                - For more information, see LVideoOut_GammaRampEntry.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetOverlayGammaRamp(
                LVideoOut_Handle            hVideoOut,
                MUINT32                     uiRampEntryCount,
                LVideoOut_GammaRampEntry*   paoRampEntries);

/************************************************************************************************************\

Function:       LVideoOut_GetOverlayGammaRamp

Description:    Gets the gamma ramp data for the video output overlay.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  uiRampEntryCount    Number of entries in the passed in ramp array.
                IN  paoRampEntries      Pointer to a user-allocated array of LVideoOut_GammaRampEntry
                                        structures where the the gamma ramp values are written.  To avoid
                                        memory overruns, this parameter must have at least the same number of
                                        entries as in uiRampEntryCount.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayGammaRamp of the LVideoOut_Caps structure is set.
                - The overlay gamma ramp is used to correct the intensity of the overlay displayed pixels.
                - If the uiRampEntryCount is different from the current gamma ramp entry count,
                  LVideoOut_GetOverlayGammaRamp fails and returns LStatus_FAIL.  To find the right entry count
                  to use, see LVideoOut_GetOverlayGammaRampInfo.
                - For more information, see LVideoOut_GammaRampEntry.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetOverlayGammaRamp(
                LVideoOut_Handle            hVideoOut,
                MUINT32                     uiRampEntryCount,
                LVideoOut_GammaRampEntry*   paoRampEntries);

/************************************************************************************************************\

Function:       LVideoOut_SetOverlayBuffer

Description:    Sets a new video overlay buffer to be displayed using the video output overlay.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  hVideoBuffer        Handle of the new video overlay buffer to display.
                IN  bIsTopField         MTRUE if the video overlay buffer to display is a top field.
                                        MFALSE if the video overlay buffer to display is a bottom field.
                IN  eFlipType           Type of flip to perform.  Must be a value from LVideoOut_FlipType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bOverlayColorKey, bOverlayAlphaBlending or 
                   bOverlayConstantBlending of the LVideoOut_Caps structure is set.
                - LVideoOut_SetOverlayBuffer overwrites any previous buffer set not yet displayed.
                - The next video overlay buffer must be the same size and format as the video overlay buffer
                  supplied when one of the following is called:
                    - LVideoOut_EnableOverlayConfigWithConstantBlendingFactor
                    - LVideoOut_EnableOverlayConfigWithPixelAlphaBlendin
                    - LVideoOut_EnableOverlayConfigWithColorKey
                - If non-interlaced video parameters are used, bIsTopField is ignored since there's no top
                  or bottom field.
                - If flip type is set to LVideoOut_FlipType_NEXT_FIELD, the flip is posted and the service
                  returns immediately.  You can use LVideoOut_WaitForFieldStart to find out when the new 
                  buffer is actually displayed.
                - If the flip type is set to LVideoOut_FlipType_NEXT_FIELD_WAIT, the flip is posted and
                  LVideoOut_SetOverlayBuffer don't return until the newly set buffer is used.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetOverlayBuffer(
                LVideoOut_Handle    hVideoOut,
                LBuffer_Handle      hVideoBuffer,
                MBOOL32             bIsTopField,
                LVideoOut_FlipType  eFlipType);

/************************************************************************************************************\

Function:       LVideoOut_ScheduleSetOverlayBuffer

Description:    Schedules an update of the displayed overlay video buffer by adding commands to hDeviceThread.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  hVideoBuffer        Handle to the video overlay buffer to set.
                IN  bIsTopField         MTRUE if the video overlay buffer to display is a top field.
                                        MFALSE if the video overlay buffer to display is a bottom field.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  eFlipType           Type of flip to perform.  Must be a value from LVideoOut_FlipType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   The function abort because its parameters are invalid.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a time out.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - For more information, see the LVideoOut_SetOverlayBuffer function.
                - LStatus_TIMEOUT is returned when the SetOverlayBuffer command couldn't be added to the 
                  device thread because it was full and no room became available before the timeout period 
                  expired.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ScheduleSetOverlayBuffer(
                LVideoOut_Handle        hVideoOut,
                LBuffer_Handle          hVideoBuffer,
                MBOOL32                 bIsTopField,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                LVideoOut_FlipType      eFlipType);


/************************************************************************************************************\

Function:       LVideoOut_ScheduleWaitOverlayFieldCounter

Description:    Schedules a wait on a overlay field counter of the video output device by adding the required
                commands to the passed in device thread object.

Parameters:     IN  hVideoOut               Handle to the video output device.
                IN  hDeviceThread           Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms            Device thread full queue wait timeout.
                IN  uiOverlayFieldCounter   Field counter value when the buffer will be displayed by
                                            the video output.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_CANCELED        Function exited because the event will never occur.
                LStatus_TIMEOUT         Function exited after a time out.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - The overlay field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.
                - LStatus_TIMEOUT is returned when the wait command couldn't be added to the device thread
                  because it was full and no room became available before the timeout period expired.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ScheduleWaitOverlayFieldCounter(
                LVideoOut_Handle        hVideoOut,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MUINT64                 uiOverlayFieldCounter);

/************************************************************************************************************\

Function:       LVideoOut_WaitForOverlayFieldStart

Description:    Blocks a CPU thread until the overlay starts its next field.

Parameters:     IN  hVideoOut               Handle of the video output object.
                IN  uiTimeout_ms            Waiting time in milliseconds.  0 means don't wait.
                                            LINFINITE_TIMEOUT means wait indefinitely.
                                            requested.
                OUT puiOverlayFieldCounter  Pointer to an MUINT64 variable where the number fields displayed
                                            by the overlay is written.  MNULL if this counter value isn't
                                            requested.
                OUT puiSetBufferCounter     Pointer to an MUINT64 variable where the current number of set 
                                            buffers performed by the video overlay is written.  MNULL if this
                                            counter value isn't requested.
                OUT puiTickRefCounter       Pointer to an MUINT64 variable where the tick reference counter
                                            value is written. This value represent the tick number at which
                                            the field counter was updated. Ticks are incremented from the 
                                            start of the system. MNULL if this counter value isn't requested.
                OUT pbIsTopField            Pointer to an MBOOL32 variable.  MTRUE if the top field is
                                            displayed.  MFALSE if the bottom field is displayed. MNULL if the
                                            displayed field isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a time out.
                LStatus_CANCELED        Function exited because it can't be completed.  (For example, the CRTC
                                        stopped running).
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bCursor of the LVideoOut_Caps structure is set.
                - If interlaced video parameters are used, use pbIsTopField to find out which of the top or
                  bottom fields are displayed.  If non-interlaced video parameters are used, pbIsTopField is
                  always set to MTRUE since there's no top or bottom field.
                - The overlay field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.
                - For more information, see LVideo_VidParam.

\************************************************************************************************************/
LAPI LStatus LVideoOut_WaitForOverlayFieldStart(
                    LVideoOut_Handle    hVideoOut,
                    MUINT32             uiTimeout_ms,
                    MUINT64*            puiOverlayFieldCounter,
                    MUINT64*            puiSetBufferCounter,
                    MUINT64*            puiTickRefCounter,
                    MBOOL32*            pbIsTopField);

/************************************************************************************************************\

Function:       LVideoOut_GetOverlayCounters

Description:    Gets the values of the set buffers and field counters of the overlay.

Parameters:     IN  hVideoOut               Handle of the video output object.
                OUT puiOverlayFieldCounter  Pointer to an MUINT64 variable where the number fields displayed
                                            by the overlay is written.  MNULL if this counter value isn't
                                            requested.
                OUT puiSetBufferCounter     Pointer to an MUINT64 variable where the current number of set 
                                            buffers performed by the video overlay is written.  MNULL if this
                                            counter value isn't requested.
                OUT puiTickRefCounter       Pointer to an MUINT64 variable where the tick reference counter
                                            value is written. This value represent the tick number at which
                                            the field counter was updated. Ticks are incremented from the 
                                            start of the system. MNULL if this counter value isn't requested.
                OUT pbIsTopField            Pointer to an MBOOL32 variable.  MTRUE if the top field is
                                            displayed.  MFALSE if the bottom field is displayed.  MNULL if the
                                            displayed field isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - The overlay field counter is incremented each time a field is displayed by the overlay.
                  The overlay set buffer counter is incremented each time a new buffer is set.
                - The overlay field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.
                - If interlaced video parameters are used, two (2) fields and two (2) set buffers are required
                  for a whole frame to be displayed.  Use pbIsTopField to find out which of the top or bottom 
                  field are displayed.  If non-interlaced video parameters are used, pbIsTopField is always 
                  set to MTRUE, since there's no top or bottom field.
                - Progressive video parameters display only one field per frame.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetOverlayCounters(
                LVideoOut_Handle    hVideoOut,
                MUINT64*            puiOverlayFieldCounter,
                MUINT64*            puiSetBufferCounter,
                MUINT64*            puiTickRefCounter,
                MBOOL32*            pbIsTopField);

/************************************************************************************************************\

Function:       LVideoOut_GetOverlayFieldCounterLUID

Description:    Gets the overlay field counter unique identifiers (LUID).

Parameters:     IN  hVideoOut                   Handle of the video output object.
                OUT puiOverlayFieldCounterLUID  Pointer to an MUINT64 variable where the overlay field
                                                counter's unique identifier is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - Use to synchronize the device thread with the completion of a counter.
                - The overlay field counter always increments by LVIDEO_FIELD_COUNTER_INCREMENT.
                - Sets to MNULL when the counter isn't supported.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetOverlayFieldCounterLUID(
                LVideoOut_Handle    hVideoOut,
                MUINT64*            puiOverlayFieldCounterLUID);

/************************************************************************************************************\

Function:       LVideoOut_SetCursorShape

Description:    Sets the shape of the cursor.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  poCursorShape       Pointer to an LVideoOut_CursorShape object containing the information
                                        of the cursor shape to set.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bCursor of the LVideoOut_Caps structure is set.
                - The cursor is invisible until LVideoOut_ShowCursor is called.  To avoid tearing when the
                  cursor is visible, the cursor shape is updated only during the next video output vertical
                  blanking interval.

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetCursorShape(
                LVideoOut_Handle        hVideoOut,
                LVideoOut_CursorShape*  poCursorShape);

/************************************************************************************************************\

Function:       LVideoOut_SetCursorPosition

Description:    Sets the hotspot position of the cursor.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  uiHotSpotXPos       Cursor hotspot position along the X axis.
                IN  uiHotSpotYPos       Cursor hotspot position along the Y axis.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bCursor of the LVideoOut_Caps structure is set.
                - The specified position is relative to the upper-left corner of the active region (0, 0).
                - Valid hotspot positions for the cursor are:
                    (0, 0) - (active region width, active region height).

Example:
                In the following configuration, the hotspot positions of the cursor are: A (0, 0),
                B (1024, 768).


                    Video parameters: 1280 x 1024
                    Active region:    1024 x 768  at position (128, 128)

                        0      128                             1152  1280
                      0 +-------+------------------------------+-------+
                        |   Video parameters                           |
                        |                                              |
                    128 +       A------------------------------+       |
                        |       |                              |       |
                        |       |                              |       |
                        |       |                              |       |
                        |       |         Active Region        |       |
                        |       |                              |       |
                        |       |                              |       |
                        |       |                              |       |
                        |       |                              |       |
                        |       |                              |       |
                    896 +       +------------------------------B       |
                        |                                              |
                        |                                              |
                   1024 +----------------------------------------------+

\************************************************************************************************************/
LAPI LStatus LVideoOut_SetCursorPosition(
                LVideoOut_Handle    hVideoOut,
                MUINT32             uiHotSpotXPos,
                MUINT32             uiHotSpotYPos);

/************************************************************************************************************\

Function:       LVideoOut_GetCursorPosition

Description:    Gets the hotspot position of the cursor.

Parameters:     IN  hVideoOut           Handle of the video output object.
                OUT puiHotSpotXPos      Pointer to an MUINT32 variable to be updated with the cursor position
                                        along the X axis.
                OUT puiHotSpotYPos      Pointer to an MUINT32 variable to be updated with the cursor position
                                        along the Y axis.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bCursor of the LVideoOut_Caps structure is set.
                - The returned position is relative to the upper-left corner of the active region.  For more
                  information, see LVideoOut_SetCursorPosition.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetCursorPosition(
                LVideoOut_Handle    hVideoOut,
                MUINT32*            puiHotSpotXPos,
                MUINT32*            puiHotSpotYPos);

/************************************************************************************************************\

Function:       LVideoOut_ShowCursor

Description:    Shows the cursor shape.

Parameters:     IN  hVideoOut           Handle of the video output object.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bCursor of the LVideoOut_Caps structure is set.
                - For a cursor to be visible, the shape information of the cursor is needed.  Call
                  LVideoOut_SetCursorShape before calling LVideoOut_ShowCursor.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ShowCursor(
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LVideoOut_HideCursor

Description:    Hides the cursor shape.

Parameters:     IN  hVideoOut       Handle of the video output object.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hVideoOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bCursor of the LVideoOut_Caps structure is set.
                - While the cursor is hidden, the shape information of the cursor is kept.  The cursor is
                  automatically hidden when LVideoOut_Disable is called.

\************************************************************************************************************/
LAPI LStatus LVideoOut_HideCursor(
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LVideoOut_GetCursorVisibility

Description:    Gets the cursor visibility

Parameters:     IN  hVideoOut       Handle of the video output object.
                OUT pbIsVisible     Pointer to an MBOOL32 variable where the cursor visibility is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       - This service is supported if bCursor of the LVideoOut_Caps structure is set.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetCursorVisibility(
                LVideoOut_Handle    hVideoOut,  
                MBOOL32*            pbIsVisible);

/************************************************************************************************************\

Function:       LVideoOut_WaitForEvent

Description:    Blocks a CPU thread until a video output event occurs.

Parameters:     IN  hVideoOut               Handle to the video output device.
                IN  uiTimeout_ms            Time out to wait in milliseconds.  0 means no wait and 
                                             LINFINITE_TIMEOUT means forever.
                OUT LVideoOut_Event         Pointer to an LVideoOut_Event object to be filled with the 
                                             signaled video output event. Valid when LStatus_OK is returned

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - We recommend creating a handle by thread to use these functions.
                - If two events are signaled before calling LVideoOut_WaitForEvent,
                  and after LVideoOut_GetHandle, LVideoIn_WaitForEvent will return two times with
                  the corresponding event code.
                - This function does not miss any events that occurred since the VideoOut handle creation.

\************************************************************************************************************/
LAPI LStatus LVideoOut_WaitForEvent(
                LVideoOut_Handle    hVideoOut,
                MUINT32             uiTimeout_ms,
                LVideoOut_Event*    peEventsSignaled);

/************************************************************************************************************\

Function:       LVideoOut_EnablePassthrough

Description:    Enable video output pass through.

Parameters:     IN  hVideoOut           Handle to the video output device.
                IN  uiVideoInIndex      Index of the independent video input for which to return the handle.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       This function is not included in LVideoOut.doc as it isn't implemented yet

\************************************************************************************************************/
LAPI LStatus LVideoOut_EnablePassthrough(
            LVideoOut_Handle        hVideoOut,
            MUINT32                 uiVideoInIndex);

/************************************************************************************************************\

Function:       LVideoOut_DisablePassthrough

Description:    Disable video output pass through.

Parameters:     IN  hVideoOut           Handle to the video output device.
                IN  uiVideoInIndex      Index of the independent video input for which to return the handle.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Function failed due to unsupported feature.

Comments:       This function is not included in LVideoOut.doc as it isn't implemented yet

\************************************************************************************************************/
LAPI LStatus LVideoOut_DisablePassthrough(
            LVideoOut_Handle        hVideoOut,
            MUINT32                 uiVideoInIndex);


/************************************************************************************************************\

Function:       LVideoOut_GetDescription

Description:    Gets the video output description.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the independent video output for which the description
                                        should be returned.
                OUT szDescription       Buffer where the ANSI null terminated video output description is
                                        written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetDescription(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                LSTR64              szDescription);

/************************************************************************************************************\

Function:       LVideoOut_GetBufferAttributes

Description:    Gets the video buffer attributes.

Parameters:     IN  hVideoOut           Handle to the video device.
                OUT pePixelFmt          Pointer to where the buffer pixel format will be written.
                OUT puiWidth            Pointer to where the buffer width will be written.
                OUT puiHeight           Pointer to where the buffer height will be written.
                OUT puiBufferLUID       Pointer to where the buffer LUID will be written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_GetBufferAttributes(
                LVideoOut_Handle        hVideoOut,
                LPixelFormat*           pePixelFmt,
                MUINT32*                puiWidth,
                MUINT32*                puiHeight,
                MUINT64*                puiBufferLUID);

/************************************************************************************************************\

Function:       LVideoOut_EnumPixelFormat

Description:    Enumerates all supported video output pixel formats.

Parameters:     IN  hVideoOut           Handle to video output device.
                IN  uiIndex             Index of the pixel format to enumerate.
                OUT pePixelFmt          Pointer to an LPixelFormat object where the enumerated pixel formats
                                        are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    End of the enumeration.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_EnumPixelFormat(
                LVideoOut_Handle        hVideoOut,
                MUINT32                 uiIndex,
                LPixelFormat*           pePixelFmt);

/************************************************************************************************************\

Function:       LVideoOut_ValidateMode

Description:    Tests the mode for a video output.

Parameters:     IN  hVideoOut           Handle of the video output object.
                IN  poBufferAttributes  Pointer to an LBuffer_VideoAttributes object containing the buffer
                                        attributes.
                IN  eSignalType         Signal type for which to validate the video parameters.
                IN  poVidParam          Pointer to an LVideo_VidParam object containing the video parameters
                                        to validate.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVideoOut_ValidateMode(
                LVideoOut_Handle            hVideoOut,
                LBuffer_VideoAttributes*    poBufferAttributes,
                LVideo_SignalType           eSignalType,
                LVideo_VidParam*            poVidParam);

#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LVIDEOOUT_H
